  clear; clc
%% Path 
addpath('Config')
addpath('Vehicle data')
addpath('R0 and COP')

%% User input
% Environment temperature
T_init = 26;
% Driving cycle
cycle = 'WLTC3';

%Target battery temperature
T_target = 30; %°C
% Target cabin temperature
T_target_HVAC = 23; %°C


% Use the backward simulator current
if strcmp(cycle,'BuildIn')
    use_simulator_current = 0;
    t = 1;
    C_rate = 1;
    t_end = 3600;
    v_vehicle = 0;
else
    use_simulator_current = 1;
    load(['../gambero/Results/BattCurrProfile/',cycle],'prof','veh');
    t = [prof.time]';
    C_rate = [prof.battCurr]'/veh.batt.nomCap;
    t_end = t(end);
    v_vehicle = [prof.vehSpd]' * 3.6; % km/h
end

% Strategy (AMPC or Reactive)
strategy = 'AMPC';
model_name = append('ElectricVehicleThermalManagement_',strategy,'_2020b_final');
open(model_name)

% Sample time for "To Workspace"
Tsample = 0.1; % set -1 to not define it 

%% Parameters to run simulation
% Load vehicle paramaters from MATWORKS example
run ElectricVehicleThermalManagementParameters.m
% Load other vehicle parameters (COP, battery thermal parameters, ecc..)
run R0_COP.m
% Initial conditions
run Initial_conditions.m
% MPC parameters
run MPC_parameters.m

% Current
I_traction = C_rate * battery_Ah_capacity;

if strcmp(strategy,'AMPC')
    % force cabin or battery priority mode
    force_sim = 1; % if 1 you force the simulation if 0 it switch automatically between cabin/battery priority mode
    controller_selector = 0; % if 0 cabin priority if 1 battery priority
    
    %% Write symbolic jacobian matrices, which means write the initial linearized model for Cabin Priority
    % Define states, outputs and inputs
    syms x [4 1] % States (x1 = battery temperature cabin_p, x2 = cabin temperature cabin_p, x3 = battery temperature batt_p, x4 = cabin temperature batt_p)
    syms y [2 1] % Output (y1 = battery temperature cabin_p, y2 = cabin temperature cabin_p)
    syms u [6 1] % Input (u1 = compressor power, u2 = current, u3 = mdot_batt_pump, u4 = Qdot_chiller, u5 = ptc_power, u6 = Qdot_cabin)  
    
    %syms thetaB SOC iB Pbtm
    syms R0 c % Define internal resistance R0 and coefficient of performance COP
    R0 = poly2sym(r_pol_coef,x1); % Cell R0 = f(battery temperature); 
    % c = c1*u3 + c0; % c is the COP: c = Qdot_BTM/Pcomp. c = f(environment temperature), 
    
    % c = 5.4;
    %c = -2.1421*(10^(-10))*u1^3+1.9594*(10^(-6))*u1^2-0.0059*u1+6.4856;

    % c_1 = -2.1421*(10^(-10))*u1^3+1.9594*(10^(-6))*u1^2-0.0060*u1+6.4856;

    % c_1 = 6.5607*10^(-7)*u1^(2)-0.0033*u1+6.1878;

    c_1 = -0.00004*u1+3.3797;
    

    % States derivative
    syms xdot [4 1] % States derivatives
    
    % m = 1.3489*10^(-4); % 1/(m_air_cab*cp)
    m = 2.7673*10^(-4);
    COP_C = c_1; % COP value

    % Cabin priority
    xdot1 = (Joule_incr_factor*battery_N_cells_per_module*battery_N_modules*R0*(u2+u1/battery_V_nom)^2 ...
    - u3*u4) / (battery_mass*battery_cell_cp);
    xdot2 = (m)*(u6+u5+u3*u4-COP_C*u1);
    
    xdot_C = [xdot1; xdot2]; 
    
    % Output
    y1 = x1;
    y2 = x2;
    
    % Jacobians Cabin
    A1s = [diff(xdot1,x1) diff(xdot1,x2);
          diff(xdot2,x1) diff(xdot2,x2)];
    B1s = [diff(xdot1,u1) diff(xdot1,u2) diff(xdot1,u3) diff(xdot1,u4) diff(xdot1,u5) diff(xdot1,u6);
          diff(xdot2,u1) diff(xdot2,u2) diff(xdot2,u3) diff(xdot2,u4) diff(xdot2,u6) diff(xdot2,u6)];
    C1s = [diff(y1,x1) diff(y1,x2); 
          diff(y2,x1) diff(y2,x2)];
    D1s = [diff(y1,u1) diff(y1,u2) diff(y1,u3) diff(y1,u4) diff(y1,u5) diff(y1,u6);
          diff(y2,u1) diff(y2,u2) diff(y2,u3) diff(y2,u4) diff(y2,u5) diff(y2,u6)];
    
    %% Symbolic function generation 
    % Creation of Jacobian, R0, and COP functions in Simulink ( block "Jacobians" (Compressor control subsystem), "R0" (interal model subsystem), "c" (interal model subsystem))
    matlabFunctionBlock(append(model_name,'/Controls/Cabin Priority/Compressor Control/Jacobians'),A1s,B1s,xdot1,xdot2)
    matlabFunctionBlock(append(model_name,'/Internal Model AMPC/R0'),R0)
    matlabFunctionBlock(append(model_name,'/Internal Model AMPC/c'),c_1)
    
    
    %% Creation of mpcobj
    % Write the initial condition x0 u0 y0
    x0 = [T_battery_init; cabin_T_init]; % look up file "ElectricVehicleManagementParamenters" initial condition block
    u1_0 = [EPowerCompressor_init; i_battery_init; mdot_pump_init; Qdot_chiller_init; EPower_ptc_init; Qdot_cabin_init];
    y0 = [T_battery_init; cabin_T_init];
    
    % Define the nominal initial point 
    x1 = sym(x0(1));
    x2 = sym(x0(2));
    u1 = sym(u1_0(1));
    u2 = sym(u1_0(2));
    u3 = sym(u1_0(3));
    u4 = sym(u1_0(4));
    u5 = sym(u1_0(5));
    u6 = sym(u1_0(6));

    % Matrixes to create the plant model Cabin Priority
    A1 = double(subs(A1s));
    B1 = double(subs(B1s));
    C1 = double(subs(C1s));
    D1 = double(subs(D1s));
    
    % Continuous and discrete time plant models Cabin Priority
    plantCT1 = ss(A1,B1,C1,D1);
    plantDT1 = c2d(plantCT1,Ts_MPC);
    % Set disturbances, manipulated variables, output
    plantDT1.InputGroup.MeasuredDisturbances = [2 4 6];
    plantDT1.InputGroup.ManipulatedVariables = [1 3 5];
    plantDT1.OutputGroup.Measured = [1 2];
    plantDT1.StateName = {'Tbattery','Tcabin'};
    plantDT1.InputName = {'Pbtm','iB','mdot_pump','Qdot_chiller','Pptc','Qdot_cabin'};
    plantDT1.OutputName = {'Tbattery','Tcabin'};
    
    % Create MPC object
    mpcobj1 = mpc(plantDT1,Ts_MPC,p,pc);
    
    % Nominal starting point
    mpcobj1.Model.Nominal = struct('X', x0, 'U', u1_0, 'Y', y0, 'DX', [0;0]);
    
    % Output variable
    % Batt temperature
    mpcobj1.OV(1).Min = min_batt_T;
    mpcobj1.OV(1).MinECR = 0; % 0 is hard constraint, the higher the softer the constraint
    mpcobj1.OV(1).Max = max_batt_T;
    mpcobj1.OV(1).MaxECR = 0; % 0 is hard constraint, the higher the softer the constraint
    mpcobj1.OV(1).ScaleFactor = T_batt_scale;
   
    % Cabin temperature
    mpcobj1.OV(2).Min = min_cabin_T;
    mpcobj1.OV(2).MinECR = 1; % 0 is hard constraint, the higher the softer the constraint
    mpcobj1.OV(2).Max = max_cabin_T;
    mpcobj1.OV(2).MaxECR = 1; % 0 is hard constraint, the higher the softer the constraint
    mpcobj1.OV(2).ScaleFactor = T_cabin_scale;
    
    % Control variable 
    % Compressor power
    mpcobj1.MV(1).Max = MaxEPowerCompressor;
    mpcobj1.MV(1).Min = 0;
    mpcobj1.MV(1).ScaleFactor = MaxEPowerCompressor;
    mpcobj1.MV(1).Target = 0; % We want that the compressor power is kept minimum
    mpcobj1.MV(1).RateMin = Pcomp_RateMin;
    mpcobj1.MV(1).RateMax = Pcomp_RateMax;
    
    % mdot_pump 
    mpcobj1.MV(2).Max = Max_mdot_pump;
    mpcobj1.MV(2).Min = (Max_mdot_pump/5); % According with previous PI controller minimum pump usage in paraller mode is 0.3
    mpcobj1.MV(2).ScaleFactor = mdot_pump_scale;
    mpcobj1.MV(2).Target = (Max_mdot_pump/5); % We want that the ptc power is kept minimum
    mpcobj1.MV(2).RateMin = mdot_pump_RateMin;
    mpcobj1.MV(2).RateMax = mdot_pump_RateMax;

    % Ptc power
    mpcobj1.MV(3).Max = MaxEPowerPtc;
    mpcobj1.MV(3).Min = 0;
    mpcobj1.MV(3).ScaleFactor = MaxEPowerPtc;
    mpcobj1.MV(3).Target = 0; % We want that the ptc power is kept minimum
    mpcobj1.MV(3).RateMin = Pptc_RateMin;
    mpcobj1.MV(3).RateMax = Pptc_RateMax;

    % Disturbance variable
    mpcobj1.DV(1).ScaleFactor = current_scale; % current 
    mpcobj1.DV(2).ScaleFactor = Qdot_chiller_scale; % Qdot_chiller/mdot_pump
    mpcobj1.DV(3).ScaleFactor = Qdot_cabin_scale; % cabin convection + heat generated by the passengers

    % Cost function weights
    mpcobj1.Weights.OutputVariables = [WeightTbattery_C*ones(p,1), WeightTcabin_C*ones(p,1)];
    mpcobj1.Weights.ManipulatedVariables = [WeightEPowerComp_C*ones(p,1), Weight_mdot_pump_C*ones(p,1), WeightEPowerPtc_C*ones(p,1)];
    mpcobj1.Weights.ManipulatedVariablesRate = [WeightEPowerRateComp_C*ones(p,1), Weight_mdot_pumpRate_C*ones(p,1), WeightEPowerRatePtc_C*ones(p,1)];
    mpcobj1.Weights.ECR = WeightECR_C;


    %% Write symbolic jacobian matrices, which means write the initial linearized model for Battery Priority
    % Define states, outputs and inputs
    syms Y [2 1] % Output (Y1 = battery temperature batt_p, Y2 = cabin temperature batt_p)
    syms U [6 1] % Input (U1 = compressor power, U2 = current, U3 = Qdot_evap, U4 = ptc_power, U5 = mdot_blower, U6 = Qdot_cabin) 

    % States derivative

    % c_2 = -2.1421*(10^(-10))*U1^3+1.9594*(10^(-6))*U1^2-0.0060*U1+6.4856;
    % c_2 = -2.1421*(10^(-10))*U1^3+2.0994*(10^(-6))*U1^2-0.0058*U1+6.4856;
    % c_2 = 5.5607*(10^(-7))*U1^2-0.0033*U1+6.1878;
    c_2 = -0.00004*U1+3.3797;
    % c_2 = 6.5607*10^(-7)*U1^(2)-0.0033*U1+6.1878;

    COP_B = c_2; % COP value

    % Battery priority
    xdot3 = (Joule_incr_factor*battery_N_cells_per_module*battery_N_modules*R0*(U2+U1/battery_V_nom)^2 ...
    - COP_B*U1+U5*U3) / (battery_mass*battery_cell_cp);
    xdot4 = (m)*(U4-U5*U3+U6);
    
    xdot_B = [xdot3; xdot4]; 
    
    % Output
    Y1 = x3;
    Y2 = x4;

    % Jacobians Battery
    A2s = [diff(xdot3,x3) diff(xdot3,x4);
          diff(xdot4,x3) diff(xdot4,x4)];
    B2s = [diff(xdot3,U1) diff(xdot3,U2) diff(xdot3,U3) diff(xdot3,U4) diff(xdot3,U5) diff(xdot3,U6);
          diff(xdot4,U1) diff(xdot4,U2) diff(xdot4,U3) diff(xdot4,U4) diff(xdot4,U5) diff(xdot4,U6)];
    C2s = [diff(Y1,x3) diff(Y1,x4); 
          diff(Y2,x3) diff(Y2,x4)];
    D2s = [diff(Y1,U1) diff(Y1,U2) diff(Y1,U3) diff(Y1,U4) diff(Y1,U5) diff(Y1,U6);
          diff(Y2,U1) diff(Y2,U2) diff(Y2,U3) diff(Y2,U4) diff(Y2,U5) diff(Y2,U6)];
    
    %% Symbolic function generation 
    % Creation of Jacobian, R0, and COP functions in Simulink ( block "Jacobians" (Compressor control subsystem), "R0" (interal model subsystem), "c" (interal model subsystem))
    matlabFunctionBlock(append(model_name,'/Controls/Battery Priority/Compressor Control/Jacobians'),A2s,B2s,xdot3,xdot4)
    
    %% Creation of mpcobj
    % Write the initial condition x0 u0 y0
    u2_0 = [EPowerCompressor_init; i_battery_init; Qdot_evap_init; EPower_ptc_init; mdot_blower_init; Qdot_cabin_init];
    
    % Define the nominal initial point 
    x3 = sym(x0(1));
    x4 = sym(x0(2));
    U1 = sym(u2_0(1));
    U2 = sym(u2_0(2));
    U3 = sym(u2_0(3));
    U4 = sym(u2_0(4));
    U5 = sym(u2_0(5));
    U6 = sym(u2_0(6));

    % Matrixes to create the plant model Battery Priority
    A2 = double(subs(A2s));
    B2 = double(subs(B2s));
    C2 = double(subs(C2s));
    D2 = double(subs(D2s));
    
    % Continuous and discrete time plant models Battery Priority
    plantCT2 = ss(A2,B2,C2,D2);
    plantDT2 = c2d(plantCT2,Ts_MPC);
    % Set disturbances, manipulated variables, output
    plantDT2.InputGroup.MeasuredDisturbances = [2 3 6];
    plantDT2.InputGroup.ManipulatedVariables = [1 4 5];
    plantDT2.OutputGroup.Measured = [1 2];
    plantDT2.StateName = {'Tbattery','Tcabin'};
    plantDT2.InputName = {'Pbtm','iB','Qdot_evap','Pptc','mdot_blower','Qdot_cabin'};
    plantDT2.OutputName = {'Tbattery','Tcabin'};
    
    % Create MPC object
    mpcobj2 = mpc(plantDT2,Ts_MPC,p,pc);
    
    % Nominal starting point
    mpcobj2.Model.Nominal = struct('X', x0, 'U', u2_0, 'Y', y0, 'DX', [0;0]);
    
    % Output variable
    % Batt temperature
    mpcobj2.OV(1).Min = min_batt_T;
    mpcobj2.OV(1).MinECR = 0; % 0 is hard constraint, the higher the softer the constraint
    mpcobj2.OV(1).Max = max_batt_T;
    mpcobj2.OV(1).MaxECR = 0; % 0 is hard constraint, the higher the softer the constraint
    mpcobj2.OV(1).ScaleFactor = T_batt_scale;
    % Cabin temperature
    mpcobj2.OV(2).Min = min_cabin_T;
    mpcobj2.OV(2).MinECR = 0; % 0 is hard constraint, the higher the softer the constraint
    mpcobj2.OV(2).Max = max_cabin_T;
    mpcobj2.OV(2).MaxECR = 0; % 0 is hard constraint, the higher the softer the constraint
    mpcobj2.OV(2).ScaleFactor = T_cabin_scale;
    
    % Control variable 
    % Compressor power
    mpcobj2.MV(1).Max = MaxEPowerCompressor;
    mpcobj2.MV(1).Min = 0;
    mpcobj2.MV(1).ScaleFactor = MaxEPowerCompressor;
    mpcobj2.MV(1).Target = 0; % We want that the compressor power is kept minimum
    mpcobj2.MV(1).RateMin = Pcomp_RateMin;
    mpcobj2.MV(1).RateMax = Pcomp_RateMax;
    
    % Ptc power
    mpcobj2.MV(2).Max = MaxEPowerPtc;
    mpcobj2.MV(2).Min = 0;
    mpcobj2.MV(2).ScaleFactor = MaxEPowerPtc;
    mpcobj2.MV(2).Target = 0; % We want that the ptc power is kept minimum
    mpcobj2.MV(2).RateMin = Pptc_RateMin;
    mpcobj2.MV(2).RateMax = Pptc_RateMax;
    
    % mdot_blower 
    mpcobj2.MV(3).Max = Max_mdot_blower;
    mpcobj2.MV(3).Min = 0;
    mpcobj2.MV(3).ScaleFactor = mdot_blower_scale;
    mpcobj2.MV(3).Target = 0.03; % We want that the ptc power is kept minimum
    mpcobj2.MV(3).RateMin = mdot_blower_RateMin;
    mpcobj2.MV(3).RateMax = mdot_blower_RateMax;

    % Disturbance variable
    mpcobj2.DV(1).ScaleFactor = current_scale; % current 
    mpcobj2.DV(2).ScaleFactor = Qdot_evap_scale; % Qdot_evap/mdot_blower
    mpcobj2.DV(3).ScaleFactor = Qdot_cabin_scale; % cabin convection + heat generated by the passengers

    % Cost function weights
    mpcobj2.Weights.OutputVariables = [WeightTbattery_B*ones(p,1), WeightTcabin_B*ones(p,1)];
    mpcobj2.Weights.ManipulatedVariables = [WeightEPowerComp_B*ones(p,1), WeightEPowerPtc_B*ones(p,1), Weight_mdot_blower_B*ones(p,1)];
    mpcobj2.Weights.ManipulatedVariablesRate = [WeightEPowerRateComp_B*ones(p,1), WeightEPowerRatePtc_B*ones(p,1), Weight_mdot_blowerRate_B*ones(p,1)];
    mpcobj2.Weights.ECR = WeightECR_B;
end


%% Run the simulation
tic 
sim(model_name) %('Prova')%
toc

%% Plot results
addpath('../Plot/')
run('../Plot/postprocessing.m')
plot_prof(time,battery,EPowers,ElectricEnergy,cabin)

%% Save results

if strcmp(strategy,'Reactive')
    save_results_Reactive(strategy,T_init,cycle,time,Distance,ElectricEnergy,EPowers, battery,cabin,cmd) %,chiller,chillerByPassValve, compressor, condenser,simlog

else
    save_results(strategy,T_init,cycle,time,Distance,ElectricEnergy,EPowers, battery,cabin,cmd,force_sim,controller_selector) %,chiller,chillerByPassValve, compressor, condenser,simlog

end